from flask import Flask, render_template, request, jsonify, send_file
import requests
import bs4
import re
import os
import tempfile
from pathlib import Path
from datetime import datetime

app = Flask(__name__)

# Security Headers Middleware
@app.after_request
def set_security_headers(response):
    """Add security headers to all responses."""
    # Prevent clickjacking attacks
    response.headers['X-Frame-Options'] = 'SAMEORIGIN'
    
    # Enable XSS protection
    response.headers['X-XSS-Protection'] = '1; mode=block'
    
    # Prevent MIME type sniffing
    response.headers['X-Content-Type-Options'] = 'nosniff'
    
    # Referrer Policy
    response.headers['Referrer-Policy'] = 'strict-origin-when-cross-origin'
    
    # Content Security Policy
    response.headers['Content-Security-Policy'] = (
        "default-src 'self'; "
        "script-src 'self' 'unsafe-inline'; "
        "style-src 'self' 'unsafe-inline'; "
        "img-src 'self' data: https:; "
        "font-src 'self' data:; "
        "connect-src 'self' https://twitsave.com https://formsubmit.co; "
        "frame-src 'none'; "
        "base-uri 'self'; "
        "form-action 'self' https://formsubmit.co;"
    )
    
    # Permissions Policy (formerly Feature Policy)
    response.headers['Permissions-Policy'] = (
        "geolocation=(), "
        "microphone=(), "
        "camera=(), "
        "payment=(), "
        "usb=(), "
        "magnetometer=(), "
        "accelerometer=(), "
        "gyroscope=()"
    )
    
    # Strict Transport Security (HSTS) - Enable when using HTTPS
    # response.headers['Strict-Transport-Security'] = 'max-age=31536000; includeSubDomains'
    
    return response

def get_video_info(url):
    """Extract video information from Twitter URL.
    
    Args:
        url (str): The twitter post URL
        
    Returns:
        dict: Contains video_url, title, and filename
    """
    try:
        api_url = f"https://twitsave.com/info?url={url}"
        response = requests.get(api_url, timeout=10)
        data = bs4.BeautifulSoup(response.text, "html.parser")
        
        download_button = data.find_all("div", class_="origin-top-right")
        if not download_button:
            return None
            
        quality_buttons = download_button[0].find_all("a")
        if not quality_buttons:
            return None
            
        highest_quality_url = quality_buttons[0].get("href")
        
        # Get video title
        title_element = data.find_all("div", class_="leading-tight")
        if title_element:
            title_p = title_element[0].find_all("p", class_="m-2")
            if title_p:
                file_name = title_p[0].text
                file_name = re.sub(r"[^a-zA-Z0-9]+", ' ', file_name).strip()
            else:
                file_name = f"twitter_video_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
        else:
            file_name = f"twitter_video_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
        
        return {
            'video_url': highest_quality_url,
            'title': file_name,
            'filename': f"{file_name}.mp4"
        }
    except Exception as e:
        print(f"Error: {e}")
        return None

@app.route('/')
def index():
    """Render the main page."""
    return render_template('index.html')

@app.route('/about')
def about():
    """Render the about page."""
    return render_template('about.html')

@app.route('/privacy')
def privacy():
    """Render the privacy policy page."""
    return render_template('privacy.html')

@app.route('/terms')
def terms():
    """Render the terms of service page."""
    return render_template('terms.html')

@app.route('/contact')
def contact():
    """Render the contact page."""
    return render_template('contact.html')

@app.route('/faq')
def faq():
    """Render the FAQ page."""
    return render_template('faq.html')

@app.route('/blog')
def blog():
    """Render the blog page."""
    return render_template('blog.html')

@app.route('/dmca')
def dmca():
    """Render the DMCA page."""
    return render_template('dmca.html')

@app.route('/sitemap.xml')
def sitemap():
    """Serve the sitemap."""
    return send_file('static/sitemap.xml', mimetype='application/xml')

@app.route('/robots.txt')
def robots():
    """Serve the robots.txt file."""
    return send_file('static/robots.txt', mimetype='text/plain')

@app.route('/api/contact', methods=['POST'])
def contact_form():
    """API endpoint to handle contact form submissions."""
    data = request.get_json()
    
    # In a production environment, you would send an email or save to a database
    # For now, we'll just log it and return success
    name = data.get('name', '')
    email = data.get('email', '')
    subject = data.get('subject', '')
    message = data.get('message', '')
    
    # Log the contact form submission
    print(f"Contact Form Submission:")
    print(f"Name: {name}")
    print(f"Email: {email}")
    print(f"Subject: {subject}")
    print(f"Message: {message}")
    
    # TODO: Implement email sending or database storage
    
    return jsonify({'message': 'Thank you for your message! We will get back to you soon.'}), 200

@app.route('/api/get-video-info', methods=['POST'])
def get_video_info_api():
    """API endpoint to get video information."""
    data = request.get_json()
    url = data.get('url', '')
    
    if not url:
        return jsonify({'error': 'No URL provided'}), 400
    
    # Basic URL validation
    if 'twitter.com' not in url and 'x.com' not in url:
        return jsonify({'error': 'Invalid Twitter/X URL'}), 400
    
    video_info = get_video_info(url)
    
    if not video_info:
        return jsonify({'error': 'Could not extract video information. Make sure the tweet contains a video.'}), 400
    
    return jsonify(video_info)

@app.route('/api/download', methods=['POST'])
def download_video():
    """API endpoint to download the video."""
    data = request.get_json()
    video_url = data.get('video_url', '')
    filename = data.get('filename', 'video.mp4')
    
    if not video_url:
        return jsonify({'error': 'No video URL provided'}), 400
    
    try:
        # Download the video to a temporary file
        response = requests.get(video_url, stream=True, timeout=30)
        response.raise_for_status()
        
        # Create a temporary file
        temp_file = tempfile.NamedTemporaryFile(delete=False, suffix='.mp4')
        
        for chunk in response.iter_content(chunk_size=8192):
            if chunk:
                temp_file.write(chunk)
        
        temp_file.close()
        
        # Send the file
        return send_file(
            temp_file.name,
            as_attachment=True,
            download_name=filename,
            mimetype='video/mp4'
        )
    except Exception as e:
        return jsonify({'error': f'Failed to download video: {str(e)}'}), 500
    finally:
        # Clean up the temporary file after a delay
        try:
            if 'temp_file' in locals():
                os.unlink(temp_file.name)
        except:
            pass

# Error handlers
@app.errorhandler(404)
def page_not_found(e):
    """Handle 404 errors."""
    return render_template('404.html'), 404

@app.errorhandler(500)
def internal_server_error(e):
    """Handle 500 errors."""
    return render_template('500.html'), 500

if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0', port=5000)
